<?php

class ABC_Stats_Tracker {
    
    private $database;
    
    public function __construct() {
        $this->database = new ABC_Stats_Database();
        
        // Add tracking script to frontend
        add_action('wp_enqueue_scripts', array($this, 'enqueue_tracking_script'));
        
        // Schedule cleanup
        add_action('abc_stats_daily_cleanup', array($this, 'cleanup_old_data'));
        if (!wp_next_scheduled('abc_stats_daily_cleanup')) {
            wp_schedule_event(time(), 'daily', 'abc_stats_daily_cleanup');
        }
    }
    
    public function enqueue_tracking_script() {
        // Track all users by default.
        // Any filtering will be handled on the client side.
        
        // Enqueue tracking script
        wp_enqueue_script(
            'abc-stats-tracker',
            ABC_STATS_PLUGIN_URL . 'assets/js/poke.js',
            array(),
            ABC_STATS_VERSION,
            true
        );
        
        // Determine page context using WordPress's query
        $context = $this->get_current_page_context();
        
        // Localize script with necessary data
        $localize_data = array(
            'apiUrl' => rest_url('abc-stats/v1/poke'),
            'context' => $context
        );
        
        // Add nonce for logged-in users to maintain authentication context
        if (is_user_logged_in()) {
            $localize_data['nonce'] = wp_create_nonce('wp_rest');
        }
        
        wp_localize_script('abc-stats-tracker', 'abcStatsTracker', $localize_data);
    }
    
    private function get_current_page_context() {
        global $wp_query;
        
        // Default context for unknown pages
        $context = array(
            'type' => 'unknown',
            'id' => null,
            'title' => wp_get_document_title()
        );
        
        // Single post or page
        if (is_singular()) {
            $post_id = get_the_ID();
            $context = array(
                'type' => get_post_type($post_id),
                'id' => $post_id,
                'title' => get_the_title($post_id)
            );
        }
        // Category archive
        elseif (is_category()) {
            $term = get_queried_object();
            $context = array(
                'type' => 'category',
                'id' => $term->term_id,
                'title' => $term->name
            );
        }
        // Tag archive
        elseif (is_tag()) {
            $term = get_queried_object();
            $context = array(
                'type' => 'post_tag',
                'id' => $term->term_id,
                'title' => $term->name
            );
        }
        // Custom taxonomy archive
        elseif (is_tax()) {
            $term = get_queried_object();
            $context = array(
                'type' => $term->taxonomy,
                'id' => $term->term_id,
                'title' => $term->name
            );
        }
        // Author archive
        elseif (is_author()) {
            $author = get_queried_object();
            $context = array(
                'type' => 'author',
                'id' => $author->ID,
                'title' => $author->display_name
            );
        }
        // Search results
        elseif (is_search()) {
            $context = array(
                'type' => 'search',
                'id' => null,
                'title' => 'Search: ' . get_search_query()
            );
        }
        // Home page (blog posts index or front page)
        elseif (is_home() || is_front_page()) {
            $front_page_id = get_option('page_on_front');
            if (is_front_page() && $front_page_id) {
                // Static front page
                $context = array(
                    'type' => 'page',
                    'id' => $front_page_id,
                    'title' => get_the_title($front_page_id)
                );
            } else {
                // Blog home page
                $context = array(
                    'type' => 'home',
                    'id' => $front_page_id ?: null,
                    'title' => get_bloginfo('name')
                );
            }
        }
        // Custom post type archive
        elseif (is_post_type_archive()) {
            $post_type = get_query_var('post_type');
            $post_type_obj = get_post_type_object($post_type);
            $context = array(
                'type' => $post_type . '_archive',
                'id' => null,
                'title' => $post_type_obj ? $post_type_obj->labels->name : $post_type
            );
        }
        // Date archives
        elseif (is_date()) {
            if (is_year()) {
                $context = array(
                    'type' => 'date_year',
                    'id' => null,
                    'title' => 'Year: ' . get_query_var('year')
                );
            } elseif (is_month()) {
                $context = array(
                    'type' => 'date_month', 
                    'id' => null,
                    'title' => 'Month: ' . get_query_var('year') . '/' . get_query_var('monthnum')
                );
            } elseif (is_day()) {
                $context = array(
                    'type' => 'date_day',
                    'id' => null,
                    'title' => 'Day: ' . get_query_var('year') . '/' . get_query_var('monthnum') . '/' . get_query_var('day')
                );
            } else {
                $context = array(
                    'type' => 'date_archive',
                    'id' => null,
                    'title' => 'Date Archive'
                );
            }
        }
        // 404 page
        elseif (is_404()) {
            $context = array(
                'type' => '404',
                'id' => null,
                'title' => 'Page Not Found'
            );
        }
        
        return $context;
    }
    
    public function track_view($url, $title, $user_role = null, $context = null) {
        // Require context - no URL-based fallbacks
        if (!$context || !is_array($context) || empty($context['type'])) {
            error_log('ABC Stats: Missing or invalid context for URL: ' . $url);
            return false;
        }
        
        // Use context directly
        $tracking_data = array(
            'content_type' => sanitize_text_field($context['type']),
            'content_id' => isset($context['id']) ? $context['id'] : null,
            'url' => sanitize_text_field($url),
            'title' => sanitize_text_field($context['title']),
            'user_role' => $user_role ? sanitize_text_field($user_role) : null,
            'view_time' => current_time('mysql', true) // true = GMT/UTC
        );
        
        // Insert into database
        return $this->database->insert_view($tracking_data);
    }
    
    public function cleanup_old_data() {
        // Get retention period from options, default to 60 days
        // 0 or negative value = unlimited retention (no cleanup)
        $retention_days = get_option('abc_stats_retention_days', 60);

        // Only run cleanup if retention period is a positive number
        // is_numeric() ensures we have a valid number (not a string like "abc")
        if (is_numeric($retention_days) && $retention_days > 0) {
            $this->database->cleanup_old_data($retention_days);
        }
    }
}
