<?php

class ABC_Stats_Database {
    
    private $table_name;
    
    public function __construct() {
        global $wpdb;
        $this->table_name = $wpdb->prefix . 'abc_stats_views';
    }
    
    public function get_table_name() {
        return $this->table_name;
    }
    
    public function create_tables() {
        global $wpdb;
        
        $charset_collate = $wpdb->get_charset_collate();
        
        $sql = "CREATE TABLE {$this->table_name} (
            id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            content_type VARCHAR(20) NOT NULL DEFAULT '',
            content_id BIGINT(20) NULL,
            url VARCHAR(500) NOT NULL DEFAULT '',
            title VARCHAR(200) NOT NULL DEFAULT '',
            user_role VARCHAR(50) NULL,
            view_time DATETIME NOT NULL,
            PRIMARY KEY (id),
            KEY idx_view_time (view_time)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }
    
    public function insert_view($data) {
        global $wpdb;
        
        $defaults = array(
            'content_type' => '',
            'content_id' => null,
            'url' => '',
            'title' => '',
            'user_role' => null,
            'view_time' => current_time('mysql', true) // true = GMT/UTC
        );
        
        $data = wp_parse_args($data, $defaults);
        
        return $wpdb->insert($this->table_name, $data);
    }
    
    public function get_recent_views($limit = 50, $offset = 0) {
        global $wpdb;
        
        return $wpdb->get_results($wpdb->prepare("
            SELECT id, content_type, content_id, url, title, user_role, view_time
            FROM {$this->table_name}
            ORDER BY view_time DESC
            LIMIT %d OFFSET %d
        ", $limit, $offset));
    }
    
    public function get_views_count() {
        global $wpdb;
        
        return $wpdb->get_var("SELECT COUNT(*) FROM {$this->table_name}");
    }
    
    public function cleanup_old_data($days = 60) {
        global $wpdb;

        // Safety check: only delete if days is a positive integer
        // This prevents accidental deletion of all data if $days = 0 or negative
        if (!is_numeric($days) || $days <= 0) {
            error_log('ABC Stats: Invalid retention days value: ' . var_export($days, true));
            return false;
        }

        return $wpdb->query($wpdb->prepare("
            DELETE FROM {$this->table_name}
            WHERE view_time < DATE_SUB(UTC_TIMESTAMP(), INTERVAL %d DAY)
        ", $days));
    }

    public function get_total_rows() {
        global $wpdb;
        
        return $wpdb->get_var("SELECT COUNT(*) FROM {$this->table_name}");
    }

	// get table size
	public function get_table_size() {
		global $wpdb;
		
		$result = $wpdb->get_row("SHOW TABLE STATUS LIKE '{$this->table_name}'");
		return $result ? $result->Data_length + $result->Index_length : 0;
	}

	// get table size for display
	public function get_table_size_for_display() {
		$size = $this->get_table_size();
		if ($size < 1024) {
			return $size . ' B';
		} elseif ($size < 1048576) {
			return round($size / 1024, 2) . ' KB';
		} elseif ($size < 1073741824) {
			return round($size / 1048576, 2) . ' MB';
		} else {
			return round($size / 1073741824, 2) . ' GB';
		}
	}
    
    public function get_views_between_timestamps($start, $end) {
        global $wpdb;

        return $wpdb->get_results($wpdb->prepare("
            SELECT
                id,
                content_type,
                content_id,
                url,
                title,
                user_role,
                view_time
            FROM {$this->table_name}
            WHERE view_time >= %s AND view_time <= %s
            ORDER BY view_time ASC
        ", $start, $end));
    }

    public function get_distinct_roles() {
        global $wpdb;

        $roles = $wpdb->get_col("
            SELECT DISTINCT user_role
            FROM {$this->table_name}
            WHERE user_role IS NOT NULL
            ORDER BY user_role ASC
        ");

        return $roles ? $roles : array();
    }
}
