<?php

class ABC_Stats_Admin {
    
    private $database;
    
    public function __construct() {
        $this->database = new ABC_Stats_Database();
        
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_scripts'));
    }
    
    public function add_admin_menu() {
        // Main dashboard page
        add_menu_page(
            __('ABC Stats', 'abc-stats'),
            __('ABC Stats', 'abc-stats'),
            'manage_options',
            'abc-stats',
            array($this, 'admin_page'),
            'dashicons-chart-bar',
            30
        );

        // Raw data submenu - Hidden for v0.2x release
        // add_submenu_page(
        //     'abc-stats',
        //     __('View Logs', 'abc-stats'),
        //     __('View Logs', 'abc-stats'),
        //     'manage_options',
        //     'abc-stats-logs',
        //     array($this, 'logs_page')
        // );
    }
    
    public function admin_page() {
        echo '<div id="abc-stats-dashboard"></div>';
    }
    
    public function logs_page() {
        $per_page = 50;
        $current_page = isset($_GET['paged']) ? max(1, intval($_GET['paged'])) : 1;
        $offset = ($current_page - 1) * $per_page;
        
        // Get data
        $logs = $this->database->get_recent_views($per_page, $offset);
        $total_items = $this->database->get_views_count();
        $total_pages = ceil($total_items / $per_page);
        
        // Display page
        ?>
        <div class="wrap">
            <h1><?php _e('View Logs', 'abc-stats'); ?></h1>
            
            <div class="abc-stats-admin-card">
                <div class="abc-stats-admin-card__header">
                    <h3><?php _e('Database Information', 'abc-stats'); ?></h3>
                </div>
                <div class="abc-stats-admin-card__body">
                    <?php
                    // Get retention settings
                    $retention_days = get_option('abc_stats_retention_days', 60);
                    $next_cleanup = wp_next_scheduled('abc_stats_daily_cleanup');
                    ?>
                    <ul>
                        <li><strong><?php _e('Total Rows:', 'abc-stats'); ?></strong> <?php echo esc_html($this->database->get_total_rows()); ?></li>
                        <li><strong><?php _e('Table Size:', 'abc-stats'); ?></strong> <?php echo esc_html($this->database->get_table_size_for_display()); ?></li>
                        <li>
                            <strong><?php _e('Retention Period:', 'abc-stats'); ?></strong>
                            <?php
                            if ($retention_days > 0) {
                                printf(_n('%d day', '%d days', $retention_days, 'abc-stats'), $retention_days);
                            } else {
                                _e('Unlimited (cleanup disabled)', 'abc-stats');
                            }
                            ?>
                        </li>
                        <li>
                            <strong><?php _e('Next Cleanup:', 'abc-stats'); ?></strong>
                            <?php
                            if ($next_cleanup) {
                                echo esc_html(gmdate('M j, Y g:i A', $next_cleanup)) . ' UTC';
                            } else {
                                _e('Not scheduled', 'abc-stats');
                            }
                            ?>
                        </li>
                    </ul>
                </div>
            </div>
            
            <div class="abc-stats-admin-card">
                <div class="abc-stats-admin-card__header">
                    <h3><?php _e('Logs Summary', 'abc-stats'); ?></h3>
                </div>
                <div class="abc-stats-admin-card__body">
                    <p><?php printf(__('Showing page %d of %d', 'abc-stats'), $current_page, $total_pages); ?></p>
                    <?php
                    // Get oldest entry timestamp
                    global $wpdb;
                    $oldest_entry = $wpdb->get_var("SELECT MIN(view_time) FROM {$this->database->get_table_name()}");
                    if ($oldest_entry) {
                        echo '<p>';
                        printf(
                            __('Oldest entry: %s UTC', 'abc-stats'),
                            '<strong>' . esc_html(gmdate('M j, Y g:i A', strtotime($oldest_entry))) . '</strong>'
                        );
                        echo '</p>';
                    }
                    ?>
                    <div class="abc-stats-logs-timezone-info">
                        <?php
                        $timezone_string = get_option('timezone_string');
                        $gmt_offset = get_option('gmt_offset');
                        
                        if ($timezone_string) {
                            $site_timezone = $timezone_string;
                        } else {
                            $offset_hours = $gmt_offset;
                            $offset_sign = $offset_hours >= 0 ? '+' : '-';
                            $offset_hours = abs($offset_hours);
                            $offset_minutes = ($offset_hours - floor($offset_hours)) * 60;
                            $site_timezone = sprintf('UTC%s%02d:%02d', $offset_sign, floor($offset_hours), $offset_minutes);
                        }
                        ?>
                        <small>
                            <?php printf(__('Times shown in UTC (database storage timezone). Your timezone: %s | Site timezone: %s', 'abc-stats'), 
                                '<code id="abc-stats-browser-tz">Loading...</code>',
                                '<code>' . esc_html($site_timezone) . '</code>'
                            ); ?>
                        </small>
                    </div>
                </div>
            </div>
            
            <?php if (empty($logs)) : ?>
                <div class="notice notice-info">
                    <p><?php _e('No tracking data found. Views will appear here once tracking begins.', 'abc-stats'); ?></p>
                </div>
            <?php else : ?>
                <table class="wp-list-table widefat fixed striped abc-stats-logs-table">
                    <thead>
                        <tr>
                            <th class="column-datetime"><?php _e('Date/Time', 'abc-stats'); ?></th>
                            <th class="column-title"><?php _e('Page Title', 'abc-stats'); ?></th>
                            <th class="column-url"><?php _e('URL', 'abc-stats'); ?></th>
                            <th class="column-type"><?php _e('Page Type', 'abc-stats'); ?></th>
                            <th class="column-role"><?php _e('User Role', 'abc-stats'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($logs as $log) : ?>
                            <tr>
                                <td class="column-datetime">
                                    <strong class="row-title"><?php echo esc_html(gmdate('M j, Y g:i A', strtotime($log->view_time))); ?> UTC</strong>
                                    <div class="row-actions mobile-meta">
                                        <span class="mobile-title"><?php echo esc_html($log->title); ?></span>
                                        <span class="mobile-separator"> | </span>
                                        <span class="mobile-type">
                                            <span class="abc-stats-page-type abc-stats-page-type-<?php echo esc_attr($log->content_type); ?>">
                                                <?php echo esc_html(ucfirst($log->content_type)); ?>
                                            </span>
                                        </span>
                                        <span class="mobile-separator"> | </span>
                                        <span class="mobile-role"><?php echo $log->user_role ? esc_html(ucfirst($log->user_role)) : '<em>' . __('Guest', 'abc-stats') . '</em>'; ?></span>
                                    </div>
                                </td>
                                <td class="column-title"><?php echo esc_html($log->title); ?></td>
                                <td class="column-url"><code><?php echo esc_html($log->url); ?></code></td>
                                <td class="column-type">
                                    <span class="abc-stats-page-type abc-stats-page-type-<?php echo esc_attr($log->content_type); ?>">
                                        <?php echo esc_html(ucfirst($log->content_type)); ?>
                                    </span>
                                </td>
                                <td class="column-role"><?php echo $log->user_role ? esc_html(ucfirst($log->user_role)) : '<em>' . __('Guest', 'abc-stats') . '</em>'; ?></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
                
                <?php if ($total_pages > 1) : ?>
                    <div class="tablenav bottom">
                        <div class="tablenav-pages">
                            <?php
                            echo paginate_links(array(
                                'base' => add_query_arg('paged', '%#%'),
                                'format' => '',
                                'current' => $current_page,
                                'total' => $total_pages,
                                'prev_text' => '&laquo;',
                                'next_text' => '&raquo;'
                            ));
                            ?>
                        </div>
                    </div>
                <?php endif; ?>
            <?php endif; ?>
        </div>
        
        <style>
        .abc-stats-logs-summary {
            background: #f0f0f1;
            padding: 10px 15px;
            border-left: 4px solid #72aee6;
            margin: 15px 0;
        }
        
        .abc-stats-logs-timezone-info {
            margin-top: 8px;
            color: #646970;
        }
        
        .abc-stats-logs-timezone-info code {
            background: #fff;
            padding: 2px 6px;
            border-radius: 3px;
            font-size: 11px;
        }
        
        .abc-stats-page-type {
            display: inline-block;
            padding: 2px 8px;
            border-radius: 3px;
            font-size: 11px;
            font-weight: bold;
            text-transform: uppercase;
        }
        .abc-stats-page-type-post { background: #e1f5fe; color: #0277bd; }
        .abc-stats-page-type-page { background: #f3e5f5; color: #7b1fa2; }
        .abc-stats-page-type-home { background: #e8f5e8; color: #2e7d32; }
        .abc-stats-page-type-category { background: #fff3e0; color: #ef6c00; }
        .abc-stats-page-type-tag { background: #fce4ec; color: #c2185b; }
        .abc-stats-page-type-author { background: #f1f8e9; color: #558b2f; }
        .abc-stats-page-type-unknown { background: #fafafa; color: #616161; }
        
        /* Mobile responsive table styles */
        .mobile-meta {
            display: none;
            margin-top: 5px;
            font-size: 13px;
        }
        
        .mobile-separator {
            color: #666;
        }
        
        /* Ensure desktop display works properly */
        @media screen and (min-width: 783px) {
            .abc-stats-logs-table .mobile-meta {
                display: none !important;
            }
            
            .abc-stats-logs-table thead {
                display: table-header-group !important;
            }
            
            .abc-stats-logs-table tbody {
                display: table-row-group !important;
            }
            
            .abc-stats-logs-table tbody tr {
                display: table-row !important;
                border: none !important;
                margin-bottom: 0 !important;
                border-radius: 0 !important;
                background: transparent !important;
            }
            
            .abc-stats-logs-table tbody td {
                display: table-cell !important;
                width: auto !important;
                padding: 8px 10px !important;
                border-bottom: 1px solid #c3c4c7 !important;
            }
            
            .abc-stats-logs-table .column-title,
            .abc-stats-logs-table .column-url,
            .abc-stats-logs-table .column-type,
            .abc-stats-logs-table .column-role,
            .abc-stats-logs-table .column-datetime {
                display: table-cell !important;
            }
        }
        
        @media screen and (max-width: 782px) {
            .abc-stats-browser-timezone-note {
                display: block;
                margin-top: 4px;
            }
            
            /* Hide columns on mobile */
            .abc-stats-logs-table .column-title,
            .abc-stats-logs-table .column-url,
            .abc-stats-logs-table .column-type,
            .abc-stats-logs-table .column-role {
                display: none !important;
            }
            
            /* Show only datetime column */
            .abc-stats-logs-table .column-datetime {
                display: table-cell !important;
                width: 100% !important;
            }
            
            /* Hide table header on mobile */
            .abc-stats-logs-table thead {
                display: none !important;
            }
            
            /* Convert table rows to cards */
            .abc-stats-logs-table tbody {
                display: block !important;
            }
            
            .abc-stats-logs-table tbody tr {
                display: block !important;
                border: 1px solid #e1e1e1 !important;
                margin-bottom: 10px !important;
                border-radius: 4px !important;
                background: #fff !important;
            }
            
            /* Convert table cells to block elements */
            .abc-stats-logs-table tbody td {
                display: block !important;
                width: 100% !important;
                text-align: left !important;
                border: none !important;
                padding: 15px !important;
                border-bottom: none !important;
            }
            
            /* Style the mobile content */
            .abc-stats-logs-table .row-title {
                font-size: 16px;
                color: #0073aa;
                font-weight: 600;
            }
            
            .abc-stats-logs-table .mobile-meta {
                display: block !important;
                line-height: 1.4;
            }
            
            .abc-stats-logs-table .mobile-title {
                font-weight: 500;
                color: #1d2327;
            }
            
            .abc-stats-logs-table .mobile-type .abc-stats-page-type {
                font-size: 10px;
            }
            
            .abc-stats-logs-table .mobile-role {
                color: #646970;
            }
        }
        
        @media screen and (max-width: 480px) {
            .abc-stats-logs-table .row-title {
                font-size: 15px;
                line-height: 1.3;
            }
            
            .abc-stats-logs-table .mobile-meta {
                font-size: 12px;
                margin-top: 8px;
            }
            
            .abc-stats-logs-table td {
                padding: 12px !important;
            }
        }
        </style>
        
        <script>
        document.addEventListener('DOMContentLoaded', function() {
            // Show browser timezone
            const browserTzElement = document.getElementById('abc-stats-browser-tz');
            if (browserTzElement) {
                try {
                    const userTimezone = Intl.DateTimeFormat().resolvedOptions().timeZone;
                    browserTzElement.textContent = userTimezone;
                } catch (e) {
                    browserTzElement.textContent = 'Unknown';
                }
            }
        });
        </script>
        <?php
    }
    
    public function enqueue_scripts($hook) {
        // Only load on our admin pages
        if ($hook !== 'toplevel_page_abc-stats' && $hook !== 'abc-stats_page_abc-stats-logs') {
            return;
        }
        
        // Load asset file generated by @wordpress/scripts
        $asset_file = ABC_STATS_PLUGIN_DIR . 'build/dashboard.asset.php';
        
        if (file_exists($asset_file)) {
            $asset = include $asset_file;
        } else {
            $asset = array(
                'dependencies' => array(
                    'wp-element',
                    'wp-components',
                    'wp-api-fetch',
                    'wp-i18n'
                ),
                'version' => ABC_STATS_VERSION
            );
        }
        
        // Enqueue built script
        wp_enqueue_script(
            'abc-stats-dashboard',
            ABC_STATS_PLUGIN_URL . 'build/dashboard.js',
            $asset['dependencies'],
            $asset['version'],
            true
        );
        
        // Enqueue styles (webpack bundles both main and component styles)
        wp_enqueue_style(
            'abc-stats-dashboard',
            ABC_STATS_PLUGIN_URL . 'build/dashboard.css',
            array('wp-components'),
            ABC_STATS_VERSION
        );
        
        // Get site timezone information
        $timezone_string = get_option('timezone_string');
        $gmt_offset = get_option('gmt_offset');
        
        // Format timezone display
        if ($timezone_string) {
            $site_timezone = $timezone_string;
        } else {
            // Handle manual UTC offset
            $offset_hours = $gmt_offset;
            $offset_sign = $offset_hours >= 0 ? '+' : '-';
            $offset_hours = abs($offset_hours);
            $offset_minutes = ($offset_hours - floor($offset_hours)) * 60;
            $site_timezone = sprintf('UTC%s%02d:%02d', $offset_sign, floor($offset_hours), $offset_minutes);
        }

        // Localize script with initial data
        wp_localize_script('abc-stats-dashboard', 'abcStatsData', array(
            'apiUrl' => rest_url('abc-stats/v1/'),
            'nonce' => wp_create_nonce('wp_rest'),
            'siteTimezone' => $site_timezone
        ));
    }
}
