<?php

class ABC_Stats_Rest_API {
    
    private $database;
    private $tracker;
    
    public function __construct() {
        $this->database = new ABC_Stats_Database();
        $this->tracker = new ABC_Stats_Tracker();
        
        add_action('rest_api_init', array($this, 'register_routes'));
    }
    
    public function register_routes() {
        // Raw views endpoint for timezone-aware processing
        register_rest_route('abc-stats/v1', '/views/raw', array(
            'methods' => 'GET',
            'callback' => array($this, 'get_raw_views'),
            'permission_callback' => array($this, 'check_admin_permissions'),
            'args' => array(
                'start' => array(
                    'required' => true,
                    'type' => 'string',
                    'format' => 'date-time',
                    'sanitize_callback' => 'sanitize_text_field'
                ),
                'end' => array(
                    'required' => true,
                    'type' => 'string',
                    'format' => 'date-time',
                    'sanitize_callback' => 'sanitize_text_field'
                )
            )
        ));
        
        // Logs endpoint for paginated log viewing
        register_rest_route('abc-stats/v1', '/logs', array(
            'methods' => 'GET',
            'callback' => array($this, 'get_logs'),
            'permission_callback' => array($this, 'check_admin_permissions'),
            'args' => array(
                'limit' => array(
                    'required' => false,
                    'type' => 'integer',
                    'default' => 10,
                    'minimum' => 1,
                    'maximum' => 100,
                    'sanitize_callback' => 'absint'
                ),
                'offset' => array(
                    'required' => false,
                    'type' => 'integer',
                    'default' => 0,
                    'minimum' => 0,
                    'sanitize_callback' => 'absint'
                )
            )
        ));

        // Database info endpoint
        register_rest_route('abc-stats/v1', '/database', array(
            'methods' => 'GET',
            'callback' => array($this, 'get_database_info'),
            'permission_callback' => array($this, 'check_admin_permissions')
        ));

        // Cleanup preflight endpoint - check how many rows would be deleted
        register_rest_route('abc-stats/v1', '/cleanup/preflight', array(
            'methods' => 'GET',
            'callback' => array($this, 'cleanup_preflight'),
            'permission_callback' => array($this, 'check_admin_permissions')
        ));

        // Manual cleanup endpoint - actually delete old data
        register_rest_route('abc-stats/v1', '/cleanup', array(
            'methods' => 'POST',
            'callback' => array($this, 'manual_cleanup'),
            'permission_callback' => array($this, 'check_admin_permissions')
        ));
        
        // Tracking endpoint
        register_rest_route('abc-stats/v1', '/poke', array(
            'methods' => 'POST',
            'callback' => array($this, 'track_view'),
            'permission_callback' => array($this, 'check_tracking_permissions'),
            'args' => array(
                'url' => array(
                    'required' => true,
                    'type' => 'string',
                    'sanitize_callback' => 'sanitize_text_field'
                ),
                'title' => array(
                    'required' => true,
                    'type' => 'string', 
                    'sanitize_callback' => 'sanitize_text_field'
                ),
                'context' => array(
                    'required' => true,
                    'type' => 'object',
                    'properties' => array(
                        'type' => array('type' => 'string'),
                        'id' => array('type' => array('integer', 'null')),
                        'title' => array('type' => 'string')
                    )
                )
            )
        ));
    }
    
    public function check_admin_permissions() {
        return current_user_can('manage_options');
    }
    
    public function check_tracking_permissions($request) {
        // For logged-in users, validate the nonce to maintain auth context
        if (is_user_logged_in()) {
            $nonce = $request->get_header('X-WP-Nonce');
            if (!$nonce || !wp_verify_nonce($nonce, 'wp_rest')) {
                return false;
            }
        }
        
        // Always allow tracking for basic view analytics
        return true;
    }
    
    public function track_view($request) {
        $url = $request->get_param('url');
        $title = $request->get_param('title');
        $context = $request->get_param('context');
        
        // Get current user role (if logged in)
        $user_role = null;
        if (is_user_logged_in()) {
            $current_user = wp_get_current_user();
            $user_role = $current_user->roles[0] ?? null;
        }
        
        // Track the view with context if provided
        $result = $this->tracker->track_view($url, $title, $user_role, $context);
        
        if ($result === false) {
            return new WP_Error('tracking_failed', 'Failed to track view', array('status' => 500));
        }
        
        return rest_ensure_response(array('success' => true));
    }
    
    public function get_raw_views($request) {
        $start = $request->get_param('start');
        $end = $request->get_param('end');

        // Validate timestamps
        if (!$start || !$end) {
            return new WP_Error('missing_timestamps', 'start and end timestamps are required', array('status' => 400));
        }

        // Get raw view data between timestamps
        $views = $this->database->get_views_between_timestamps($start, $end);

        // Get currently defined WordPress roles
        $wp_roles = wp_roles();
        $current_roles = array_keys($wp_roles->roles);

        // Get historical roles from database (may include deleted roles)
        $historical_roles = $this->database->get_distinct_roles();

        // Merge current and historical roles, remove duplicates
        $available_roles = array_unique(array_merge($current_roles, $historical_roles));

        // Sort alphabetically
        sort($available_roles);

        return rest_ensure_response(array(
            'views' => $views,
            'start' => $start,
            'end' => $end,
            'available_roles' => $available_roles
        ));
    }
    
    public function get_logs($request) {
        $limit = $request->get_param('limit') ?: 10;
        $offset = $request->get_param('offset') ?: 0;

        // Get recent views with pagination
        $logs = $this->database->get_recent_views($limit, $offset);

        return rest_ensure_response(array(
            'views' => $logs,
            'limit' => $limit,
            'offset' => $offset
        ));
    }

    public function get_database_info($request) {
        global $wpdb;

        // Get database stats
        $total_rows = $this->database->get_total_rows();
        $table_size = $this->database->get_table_size_for_display();

        // Get oldest entry
        $oldest_entry = $wpdb->get_var("
            SELECT view_time
            FROM {$this->database->get_table_name()}
            ORDER BY view_time ASC
            LIMIT 1
        ");

        // Get retention period from options
        $retention_days = get_option('abc_stats_retention_days', 60);

        // Get next scheduled cleanup
        $next_cleanup = wp_next_scheduled('abc_stats_daily_cleanup');

        // Check cron status (is the cleanup job scheduled?)
        $cron_status = $next_cleanup !== false;

        return rest_ensure_response(array(
            'total_rows' => (int) $total_rows,
            'table_size' => $table_size,
            'oldest_entry' => $oldest_entry, // MySQL datetime in UTC
            'retention_days' => (int) $retention_days,
            'next_cleanup' => $next_cleanup ? gmdate('Y-m-d H:i:s', $next_cleanup) : null, // Convert to MySQL datetime format
            'cron_status' => $cron_status
        ));
    }

    public function cleanup_preflight($request) {
        global $wpdb;

        // Get retention period from options
        $retention_days = get_option('abc_stats_retention_days', 60);

        // Safety check
        if (!is_numeric($retention_days) || $retention_days <= 0) {
            return new WP_Error('invalid_retention', 'Invalid retention period configured', array('status' => 400));
        }

        // Count rows that would be deleted
        $rows_to_delete = $wpdb->get_var($wpdb->prepare("
            SELECT COUNT(*)
            FROM {$this->database->get_table_name()}
            WHERE view_time < DATE_SUB(UTC_TIMESTAMP(), INTERVAL %d DAY)
        ", $retention_days));

        // Get the cutoff date for display
        $cutoff_date = $wpdb->get_var($wpdb->prepare("
            SELECT DATE_SUB(UTC_TIMESTAMP(), INTERVAL %d DAY) as cutoff
        ", $retention_days));

        return rest_ensure_response(array(
            'rows_to_delete' => (int) $rows_to_delete,
            'retention_days' => (int) $retention_days,
            'cutoff_date' => $cutoff_date // MySQL datetime in UTC
        ));
    }

    public function manual_cleanup($request) {
        global $wpdb;

        // Get retention period from options
        $retention_days = get_option('abc_stats_retention_days', 60);

        // Run the cleanup
        $rows_deleted = $this->database->cleanup_old_data($retention_days);

        if ($rows_deleted === false) {
            return new WP_Error('cleanup_failed', 'Cleanup failed', array('status' => 500));
        }

        // Get updated database info
        $total_rows = $this->database->get_total_rows();
        $table_size = $this->database->get_table_size_for_display();

        return rest_ensure_response(array(
            'success' => true,
            'rows_deleted' => (int) $rows_deleted,
            'total_rows' => (int) $total_rows,
            'table_size' => $table_size
        ));
    }
}
